<?php

namespace validation\rules {
	
	abstract class AbstractRule {
		
		/**
		 * Name of the field that will hold the value for validation
		 * @var string 
		 */
		private $field;
		/**
		 * Message to display to the user when some error occurs
		 * @var string 
		 */
		private $errorMessage;
		
		/**
		 * Execute this rule.
		 * 
		 * All classes must implement this method and 
		 * it must return a boolean value (TRUE or FALSE).
		 * 
		 * @param object $obj Object that will be validated
		 * @author Hugo Ferreira da Silva
		 * @return boolean
		 */
		abstract public function execute($obj);
		
		/**
		 * Constructor
		 * 
		 * @param string $field
		 * @param string $errorMessage
		 * @author Hugo Ferreira da Silva
		 */
		public function __construct($field, $errorMessage){
			$this->setField($field);
			$this->setErrorMessage($errorMessage);
		}
		
		/**
		 * 
		 * @author Hugo Ferreira da Silva
		 * @return string
		 */
		public function getField(){
			return $this->field;
		}
		
		/**
		 * 
		 * @param string $field
		 * @author Hugo Ferreira da Silva
		 * @return void
		 */
		public function setField($field){
			$this->field = $field;
		}
		
		/**
		 * @param string $errorMessage
		 * @author Hugo Ferreira da Silva
		 * @return void
		 */
		public function setErrorMessage($errorMessage){
			$this->errorMessage = $errorMessage;
		}
		
		/**
		 * @author Hugo Ferreira da Silva
		 * @return string 
		 */
		public function getErrorMessage(){
			return $this->errorMessage;
		}
		
		/**
		 * Retrieves the value from object field
		 * 
		 * @param object $obj
		 * @author Hugo Ferreira da Silva
		 * @return mixed
		 */
		protected function getFieldValue($obj){
			$field = $this->getField();
			$reflection = new \ReflectionClass($obj);
			
			$getMethod = null;
			
			if($reflection->hasMethod('get'.ucfirst($field))){
				$getMethod = $reflection->getMethod('get'.ucfirst($field));
			}
		
			if($getMethod && $getMethod->isPublic()){
				$value = $getMethod->invoke($obj);
			} else {
				$reflectionField = $reflection->getProperty($field);
				
				if(!$reflectionField->isPublic()){
					$reflectionField->setAccessible(true);
				}
				
				$value = $reflectionField->getValue($obj);
				
				if(!$reflectionField->isPublic()){
					$reflectionField->setAccessible(false);
				}
			}
			
			return $value;
		}
	}
	
}



/* End of file AbstractRule.php */
/* Location: AbstractRule.php */